<?php
session_start();

// === 1. 读取 .env（现在在 feedback/ 目录下）===
$envPath = __DIR__ . '/.env';
if (!is_readable($envPath))
  die('错误：无法读取 .env 文件（请检查 feedback/.env 是否存在）');

$lines = file($envPath, FILE_IGNORE_NEW_LINES | FILE_SKIP_EMPTY_LINES);
$env = [];
foreach ($lines as $line) {
  if (strpos(trim($line), '#') === 0)
    continue;
  list($key, $value) = explode('=', $line, 2);
  $env[trim($key)] = trim($value);
}

$BOT_TOKEN = $env['BOT_TOKEN'] ?? '';
$ADMIN_CHAT_IDS = array_filter(array_map('trim', explode(',', $env['ADMIN_CHAT_ID'] ?? '')));
$MAX_CHARS_FRONT = 512;
$MAX_CHARS_BACK = 1024;

if (empty($BOT_TOKEN) || empty($ADMIN_CHAT_IDS)) {
  die('错误：BOT_TOKEN 或 ADMIN_CHAT_ID 为空');
}

// === 2. IP 限流 ===
$ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
$limit_key = "feedback_limit_{$ip}";
$cooldown = 60;

if (isset($_SESSION[$limit_key]) && (time() - $_SESSION[$limit_key] < $cooldown)) {
  $wait = $cooldown - (time() - $_SESSION[$limit_key]);
  if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['success' => false, 'message' => "请 {$wait} 秒后再试"]);
    exit;
  }
}

// === 3. 验证码 ===
if (isset($_GET['captcha'])) {
  header('Cache-Control: no-cache');
  $code = sprintf('%04d', rand(0, 9999));
  $_SESSION['captcha'] = $code;

  $im = imagecreatetruecolor(120, 40);
  $bg = imagecolorallocate($im, 255, 255, 255);
  $text = imagecolorallocate($im, 0, 0, 0);
  $noise = imagecolorallocate($im, 180, 180, 180);

  imagefilledrectangle($im, 0, 0, 120, 40, $bg);
  for ($i = 0; $i < 6; $i++) {
    imageline($im, rand(0, 120), rand(0, 40), rand(0, 120), rand(0, 40), $noise);
  }
  for ($i = 0; $i < 100; $i++) {
    imagesetpixel($im, rand(0, 120), rand(0, 40), $noise);
  }

  $x = 15;
  for ($i = 0; $i < 4; $i++) {
    imagestring($im, 5, $x, 10, $code[$i], $text);
    $x += 25 + rand(0, 5);
  }

  header('Content-Type: image/png');
  imagepng($im);
  imagedestroy($im);
  exit;
}

// === 4. POST 处理 ===
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $name = trim($_POST['name'] ?? '');
  $type = trim($_POST['type'] ?? '');
  $message = trim($_POST['message'] ?? '');
  $user_captcha = $_POST['captcha'] ?? '';

  $typeMap = [
    'bug' => '问题反馈',
    'suggest' => '改进建议',
    'recommend_movie' => '电影推荐',
    'recommend_book' => '书籍推荐',
    'recommend_software' => '软件推荐',
    'recommend_website' => '网站推荐',
    'recommend_av' => '番号推荐',
    'recommend_other' => '其他推荐',
    'other' => '其他投稿',
    'whisper' => '悄悄话'
  ];
  $displayType = $typeMap[$type] ?? '未分类';

  if (empty($message)) {
    $result = ['success' => false, 'message' => '留言不能为空'];
  } elseif (strlen($message) > $MAX_CHARS_BACK) {
    $result = ['success' => false, 'message' => '留言过长，请控制在 512 字以内'];
  } elseif (empty($user_captcha) || $user_captcha !== ($_SESSION['captcha'] ?? '')) {
    $result = ['success' => false, 'message' => '验证码错误'];
  } else {
    $_SESSION[$limit_key] = time();

    $displayName = $name ?: '匿名用户';
    $submitTime = date('Y/m/d H:i:s');

    // === 构建消息（可选显示属地）===
    $text = "<b>用户昵称：{$displayName}</b>\n";
    $text .= "<b>留言类型：</b> #{$displayType}\n\n";
    $text .= "<b>留言内容：</b>\n";
    $text .= htmlspecialchars($message);
    $text .= "\n\n";
    $text .= "<b>提交时间：</b> {$submitTime}";

    // === 发送到多个 Chat ID ===
    foreach ($ADMIN_CHAT_IDS as $chatId) {
      $ch = curl_init();
      curl_setopt_array($ch, [
        CURLOPT_URL => "https://api.telegram.org/bot{$BOT_TOKEN}/sendMessage",
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => [
          'chat_id' => $chatId,
          'text' => $text,
          'parse_mode' => 'HTML'
        ],
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 10
      ]);
      curl_exec($ch);
      curl_close($ch);
    }

    $result = ['success' => true, 'message' => '留言已送达！'];
    unset($_SESSION['captcha']);
  }

  header('Content-Type: application/json; charset=utf-8');
  echo json_encode($result);
  exit;
}
?>

<!DOCTYPE html>
<html lang="zh-CN">

<head>
  <meta charset="UTF-8" />
  <meta name="viewport" content="width=device-width, initial-scale=1.0" />
  <title>匿名留言 - 毒奶 - 欢迎使用代理访问本站。</title>
  <link rel="stylesheet" href="style.css">
  <style>
    .input-group {
      margin-bottom: 15px;
    }

    textarea,
    input[type="text"],
    select {
      width: 100%;
      padding: 10px;
      border: 1px solid #ddd;
      border-radius: 6px;
      font-size: 14px;
      box-sizing: border-box;
    }

    textarea {
      resize: vertical;
    }

    select {
      background: white;
      cursor: pointer;
    }

    input[type="text"]:focus,
    textarea:focus,
    select:focus {
      outline: none;
      border-color: #1a73e8;
      box-shadow: 0 0 0 2px rgba(26, 115, 232, 0.15);
    }

    .paste-wrapper {
      display: flex;
      justify-content: flex-start;
      align-items: center;
      gap: 10px;
      margin-top: 6px;
      flex-wrap: wrap;
    }

    .paste-btn,
    .clear-btn,
    .back-btn,
    .random-nick-btn {
      background: #1a73e8;
      color: white;
      border: none;
      padding: 6px 12px;
      border-radius: 4px;
      font-size: 13px;
      font-weight: 500;
      cursor: pointer;
      white-space: nowrap;
    }

    .paste-btn:hover {
      background: #0d47a1;
    }

    .clear-btn {
      background: #f44336;
    }

    .clear-btn:hover {
      background: #d32f2f;
    }

    .back-btn {
      text-decoration: none;
      background: #f1f3f4;
      color: #333;
      border: 1px solid #ddd;
      padding: 6px 12px;
      font-size: 13px;
    }

    .back-btn:hover {
      background: #e0e0e0;
    }

    .random-nick-btn {
      background: #9c27b0;
      padding: 6px 10px;
      font-size: 12px;
    }

    .random-nick-btn:hover {
      background: #7b1fa2;
    }

    .captcha-group {
      display: flex;
      align-items: center;
      gap: 10px;
      margin: 15px 0;
    }

    .captcha-img {
      cursor: pointer;
      border: 1px solid #ddd;
      border-radius: 6px;
      height: 40px;
    }

    .captcha-input {
      width: 100px;
      padding: 8px;
      font-size: 16px;
    }

    .refresh-btn {
      cursor: pointer;
      color: #1a73e8;
      font-size: 14px;
    }

    .char-counter {
      font-size: 12px;
      color: #666;
      text-align: right;
      margin-top: 4px;
    }

    .char-counter.warning {
      color: #ff9800;
    }

    .char-counter.danger {
      color: #f44336;
    }
  </style>
</head>

<body>
  <div class="container">
    <div class="card">
      <h1>匿名留言板</h1>

      <form id="feedbackForm">
        <!-- 昵称 + 随机按钮 + 历史下拉 -->
        <div class="input-group">
          <label for="name">昵称</label>
          <div style="display: flex; gap: 8px; align-items: center; flex-wrap: wrap;">
            <input type="text" id="name" name="name" placeholder="如：路人甲" style="flex: 1; min-width: 60%;">
            <button type="button" class="random-nick-btn" onclick="generateNickname()">生成随机昵称</button>
            <select id="nickHistory" onchange="selectNickname(this.value)"
              style="padding: 6px; font-size: 12px; border-radius: 4px; border: 1px solid #ddd;">
              <option value="">已用昵称</option>
            </select>
          </div>
        </div>

        <!-- 留言类型 -->
        <div class="input-group">
          <label for="type">留言类型 <span style="color:red">*</span></label>
          <select id="type" name="type" required>
            <option value="" disabled selected>请选择类型</option>
            <optgroup label="反馈类">
              <option value="bug">问题反馈</option>
              <option value="suggest">改进建议</option>
              <option value="other">其他投稿</option>
              <option value="whisper">悄悄话</option>
            </optgroup>
            <optgroup label="推荐类">
              <option value="recommend_movie">电影推荐</option>
              <option value="recommend_book">书籍推荐</option>
              <option value="recommend_software">软件推荐</option>
              <option value="recommend_website">网站推荐</option>
              <option value="recommend_av">番号推荐</option>
              <option value="recommend_other">其他推荐</option>
            </optgroup>
          </select>
        </div>

        <!-- 留言内容 + 按钮组 -->
        <div class="input-group">
          <label for="message">留言内容 <span style="color:red">*</span></label>
          <textarea id="message" name="message" placeholder="写下你的留言..." rows="4" required maxlength="512"></textarea>
          <div class="char-counter" id="charCounter">已输入 0/512</div>
          <div class="paste-wrapper">
            <button type="button" class="paste-btn" onclick="pasteToMessage()">粘贴</button>
            <button type="button" class="clear-btn" onclick="clearMessage()">清空</button>
            <button type="button" class="back-btn" onclick="showHistoryModal()">已提交的留言</button>
            <a href="https://limbopro.com/6.html" target="_blank" class="back-btn">联系博主</a>
            <a href="https://limbopro.com/" target="_blank" class="back-btn">返回博客</a>
          </div>
        </div>

        <!-- 验证码 -->
        <div class="captcha-group">
          <img id="captchaImg" src="?captcha=1&t=<?= time() ?>" alt="验证码" class="captcha-img" />
          <input type="text" name="captcha" class="captcha-input" placeholder="验证码" maxlength="4" required />
          <span class="refresh-btn" onclick="refreshCaptcha()">刷新</span>
        </div>

        <button type="submit" id="submitBtn">发送匿名留言</button>
      </form>

      <!-- 历史记录弹窗 -->
      <div id="historyModal"
        style="display: none; position: fixed; top: 0; left: 0; width: 100%; height: 100%; background: rgba(0,0,0,0.5); z-index: 999; justify-content: center; align-items: center;">
        <div
          style="background: white; width: 90%; max-width: 500px; max-height: 70vh; border-radius: 12px; overflow: hidden; box-shadow: 0 4px 20px rgba(0,0,0,0.2);">
          <div
            style="padding: 16px; border-bottom: 1px solid #eee; display: flex; justify-content: space-between; align-items: center; font-weight: 600;">
            <span>已提交的留言（最近10条）</span>
            <button onclick="closeHistoryModal()"
              style="background:none; border:none; font-size:18px; cursor:pointer; color:#666;">X</button>
          </div>
          <div id="historyList" style="padding: 16px; max-height: 50vh; overflow-y: auto; font-size: 14px;"></div>
          <div style="padding: 12px; text-align: center; border-top: 1px solid #eee; color: #999; font-size: 12px;">
            仅显示当前浏览器记录
          </div>
        </div>
      </div>

      <div id="status"></div>
    </div>
  </div>

  <script>
    const MAX_CHARS = 512;
    const textarea = document.getElementById('message');
    const counter = document.getElementById('charCounter');
    const submitBtn = document.getElementById('submitBtn');

    // === 页面加载时恢复昵称 ===
    window.addEventListener('load', () => {
      const savedNickname = localStorage.getItem('feedback_nickname');
      if (savedNickname) {
        document.getElementById('name').value = savedNickname;
      }
      updateCharCount();
      loadNicknameHistory();
    });

    // === 输入框变化时同步 localStorage ===
    document.getElementById('name').addEventListener('input', function () {
      const value = this.value.trim();
      if (value === '') {
        localStorage.removeItem('feedback_nickname');
      } else {
        localStorage.setItem('feedback_nickname', value);
      }
    });

    // === 历史昵称管理 ===
    const NICK_HISTORY_KEY = 'feedback_nick_history';
    const MAX_HISTORY = 10;

    function loadNicknameHistory() {
      const history = JSON.parse(localStorage.getItem(NICK_HISTORY_KEY) || '[]');
      const select = document.getElementById('nickHistory');
      select.innerHTML = '<option value="">已用昵称</option>';
      history.forEach(nick => {
        const opt = document.createElement('option');
        opt.value = nick;
        opt.textContent = nick;
        select.appendChild(opt);
      });
    }

    function addToNicknameHistory(nickname) {
      if (!nickname.trim()) return;
      let history = JSON.parse(localStorage.getItem(NICK_HISTORY_KEY) || '[]');
      history = history.filter(n => n !== nickname);
      history.unshift(nickname);
      if (history.length > MAX_HISTORY) history = history.slice(0, MAX_HISTORY);
      localStorage.setItem(NICK_HISTORY_KEY, JSON.stringify(history));
      loadNicknameHistory();
    }

    function selectNickname(nickname) {
      if (nickname) {
        const input = document.getElementById('name');
        input.value = nickname;
        localStorage.setItem('feedback_nickname', nickname);
      }
    }

    // === 已提交留言历史 ===
    const MSG_HISTORY_KEY = 'feedback_msg_history';
    const MAX_MSG_HISTORY = 10;

    function addToMessageHistory(nickname, message) {
      if (!message.trim()) return;
      const timestamp = new Date().toLocaleString('zh-CN', {
        year: 'numeric',
        month: '2-digit',
        day: '2-digit',
        hour: '2-digit',
        minute: '2-digit',
        second: '2-digit'
      });
      let history = JSON.parse(localStorage.getItem(MSG_HISTORY_KEY) || '[]');
      history.unshift({ nickname, message, timestamp });
      if (history.length > MAX_MSG_HISTORY) history = history.slice(0, MAX_MSG_HISTORY);
      localStorage.setItem(MSG_HISTORY_KEY, JSON.stringify(history));
    }

    function showHistoryModal() {
      const history = JSON.parse(localStorage.getItem(MSG_HISTORY_KEY) || '[]');
      const list = document.getElementById('historyList');
      if (history.length === 0) {
        list.innerHTML = '<div style="text-align:center; color:#999; padding:20px;">暂无记录</div>';
      } else {
        list.innerHTML = history.map(item => `
          <div style="margin-bottom:16px; padding-bottom:12px; border-bottom:1px dashed #eee;">
            <div style="font-weight:600; color:#1a73e8; margin-bottom:4px;">
              ${escapeHtml(item.nickname)} <span style="font-weight:normal; color:#999; font-size:12px;">${item.timestamp}</span>
            </div>
            <div style="white-space: pre-wrap; word-break: break-word; color:#333;">${escapeHtml(item.message)}</div>
          </div>
        `).join('');
      }
      document.getElementById('historyModal').style.display = 'flex';
    }

    function closeHistoryModal() {
      document.getElementById('historyModal').style.display = 'none';
    }

    function escapeHtml(text) {
      const div = document.createElement('div');
      div.textContent = text;
      return div.innerHTML;
    }

    function updateCharCount() {
      const len = textarea.value.length;
      counter.textContent = `已输入 ${len}/${MAX_CHARS}`;
      counter.classList.remove('warning', 'danger');
      if (len > MAX_CHARS * 0.9) counter.classList.add('warning');
      if (len >= MAX_CHARS) counter.classList.add('danger');
      submitBtn.disabled = len === 0 || len > MAX_CHARS;
    }

    textarea.addEventListener('input', updateCharCount);

    async function pasteToMessage() {
      try {
        const text = await navigator.clipboard.readText();
        const current = textarea.value;
        const combined = current + (current && !current.endsWith('\n') ? '\n' : '') + text;
        if (combined.length > MAX_CHARS) {
          textarea.value = combined.substring(0, MAX_CHARS);
          showStatus('内容过长，已自动截断', 'warning');
        } else {
          textarea.value = combined;
        }
        updateCharCount();
        showStatus('已粘贴！', 'success');
      } catch (err) {
        showStatus('粘贴失败（需 HTTPS）', 'error');
      }
    }

    function clearMessage() {
      textarea.value = '';
      updateCharCount();
      showStatus('已清空！', 'success');
    }

    function refreshCaptcha() {
      document.getElementById('captchaImg').src = '?captcha=1&t=' + Date.now();
    }

    // === 随机昵称生成器 ===
    async function generateNickname() {
      try {
        const res = await fetch('./nicknames.json?t=' + Date.now());
        if (!res.ok) throw new Error(`HTTP ${res.status}`);
        const data = await res.json();

        const animal = data.animals[Math.floor(Math.random() * data.animals.length)];
        const action = data.actions[Math.floor(Math.random() * data.actions.length)];
        const place = data.places[Math.floor(Math.random() * data.places.length)];

        const nickname = place + action + animal;
        const input = document.getElementById('name');
        input.value = nickname;

        localStorage.setItem('feedback_nickname', nickname);
        addToNicknameHistory(nickname);
      } catch (err) {
        console.error('Nickname Error:', err);
        showStatus('生成失败，请检查文件', 'error');
      }
    }

    document.getElementById('feedbackForm').addEventListener('submit', async (e) => {
      e.preventDefault();
      const formData = new FormData(e.target);

      if (!e.target.type.value) {
        showStatus('请选择留言类型', 'error');
        return;
      }
      if (!e.target.message.value.trim()) {
        showStatus('请输入留言', 'error');
        return;
      }
      if (e.target.message.value.length > MAX_CHARS) {
        showStatus('留言不得超过 512 字', 'error');
        return;
      }
      if (!e.target.captcha.value) {
        showStatus('请输入验证码', 'error');
        return;
      }

      showStatus('发送中...', 'loading');

      try {
        const res = await fetch('', {
          method: 'POST',
          body: new URLSearchParams(formData)
        });
        const data = await res.json();
        showStatus(data.message, data.success ? 'success' : 'error');
        if (data.success) {
          const currentNick = document.getElementById('name').value.trim() || '匿名用户';
          const currentMsg = e.target.message.value.trim();
          if (currentMsg) {
            addToMessageHistory(currentNick, currentMsg);
            addToNicknameHistory(currentNick);
          }

          e.target.message.value = '';
          e.target.type.value = '';
          e.target.captcha.value = '';

          updateCharCount();
          refreshCaptcha();
          showStatus('留言已送达！', 'success');
        }
      } catch {
        showStatus('网络错误', 'error');
      }
    });

    function showStatus(msg, type) {
      const el = document.getElementById('status');
      el.textContent = msg;
      el.className = type;
      if (window.statusTimeout) clearTimeout(window.statusTimeout);
      window.statusTimeout = setTimeout(() => {
        el.textContent = '';
        el.className = '';
      }, 2500);
    }
  </script>
</body>

</html>
